/*
 *  $Id: netutilities.c,v 1.10 2002/08/22 09:37:23 dreibh Exp $
 *
 * RSerPool implementation.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de/rserpool.html
 * which should be used for any discussion related to this implementation.
 *
 * Contact: rsplib-discussion@sctp.de
 *          dreibh@exp-math.uni-essen.de
 *
 * Purpose: Network Utilities
 *
 */


#include "tdtypes.h"
#include "utilities.h"
#include "netutilities.h"
#include "randomizer.h"

#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/ioctl.h>
#include <net/if.h>
#include <fcntl.h>

#include <ext_socket.h>
#include <sys/uio.h>



#define MAX_AUTOSELECT_TRIALS 50000
#define MIN_AUTOSELECT_PORT   10000
#define MAX_AUTOSELECT_PORT   60000



/* ###### Get local addresses from socket ################################ */
size_t getAddressesFromSocket(int sockfd, struct sockaddr_storage** addressArray)
{
   struct sockaddr_storage address;
   socklen_t               addressLength;
   ssize_t                 addresses;
   ssize_t                 result;

   LOG_VERBOSE4
   fputs("Getting transport addresses from socket...\n",stdlog);
   LOG_END

   addresses = sctp_getladdrs(sockfd,0,addressArray);
   if(addresses < 1) {
      LOG_VERBOSE4
      logerror("sctp_getladdrs() failed, trying getsockname()");
      LOG_END

      addresses     = 0;
      *addressArray = NULL;
      addressLength = sizeof(address);
      result = ext_getsockname(sockfd,(struct sockaddr*)&address,&addressLength);
      if(result == 0) {
         LOG_VERBOSE4
         fputs("Successfully obtained address by getsockname()\n",stdlog);
         LOG_END

         *addressArray = duplicateAddressArray((const struct sockaddr_storage*)&address,1);
         if(*addressArray != NULL) {
            addresses = 1;
         }
      }
      else {
         LOG_VERBOSE4
         logerror("getsockname() failed");
         LOG_END
      }
   }
   else {
      LOG_VERBOSE4
      fprintf(stdlog,"Obtained %d address(es)\n",addresses);
      LOG_END
   }

   return((size_t)addresses);
}


/* ###### Delete address array ########################################### */
void deleteAddressArray(struct sockaddr_storage* addressArray)
{
   if(addressArray != NULL) {
      free(addressArray);
   }
}


/* ###### Duplicate address array ######################################## */
struct sockaddr_storage* duplicateAddressArray(const struct sockaddr_storage* addressArray,
                                               const size_t                   addresses)
{
   const size_t size = sizeof(struct sockaddr_storage) * addresses;

   struct sockaddr_storage* copy = (struct sockaddr_storage*)malloc(size);
   if(copy != NULL) {
      memcpy(copy,addressArray,size);
   }
   return(copy);
}


/* ###### Convert address to string ###################################### */
bool address2string(const struct sockaddr* address, char* buffer, const size_t length, const bool port)
{
   struct sockaddr_in*  ipv4address;
#ifdef HAVE_IPV6
   struct sockaddr_in6* ipv6address;
   char                 str[128];
#endif

   switch(address->sa_family) {
      case AF_INET:
         ipv4address = (struct sockaddr_in*)address;
         if(port) {
            snprintf(buffer, length,
                     "%s:%d", inet_ntoa(ipv4address->sin_addr), ntohs(ipv4address->sin_port));
         }
         else {
            snprintf(buffer, length, "%s", inet_ntoa(ipv4address->sin_addr));
         }
         return(true);
       break;
#ifdef HAVE_IPV6
      case AF_INET6:
         ipv6address = (struct sockaddr_in6*)address;
         if(inet_ntop(AF_INET6,&ipv6address->sin6_addr, str, sizeof(str)) != NULL) {
            if(port) {
               snprintf(buffer, length,
                        "%s:%d", str, ntohs(ipv6address->sin6_port));
            }
            else {
               snprintf(buffer, length, "%s", str);
            }
            return(true);
         }
       break;
#endif
      case AF_UNSPEC:
         safestrcpy(buffer,"(unspecified)",length);
         return(true);
       break;
   }

   snprintf(buffer,length,"(unsupported address family #%d)",address->sa_family);
   return(false);
}


/* ###### Convert string to address ###################################### */
bool string2address(const char* string, struct sockaddr_storage* address)
{
   char                 host[128];
   char                 port[128];
   struct sockaddr_in*  ipv4address = (struct sockaddr_in*)address;
#ifdef HAVE_IPV6
   struct sockaddr_in6* ipv6address = (struct sockaddr_in6*)address;
#endif
   char*                p1;
   int                  portNumber;

   struct addrinfo  hints;
   struct addrinfo* res;
   bool isNumeric;
   bool isIPv6;
   size_t hostLength;
   size_t i;



   if(strlen(string) > sizeof(host)) {
      LOG_ERROR
      fputs("String too long!\n",stdlog);
      LOG_END
      return(false);
   }
   strcpy((char*)&host,string);
   strcpy((char*)&port,"0");

   /* ====== Handle RFC2732-compliant addresses ========================== */
   if(string[0] == '[') {
      p1 = strindex(host,']');
      if(p1 != NULL) {
         if((p1[1] == ':') || (p1[1] == '!')) {
            strcpy((char*)&port,&p1[2]);
         }
         strncpy((char*)&host,(char*)&host[1],(long)p1 - (long)host - 1);
         host[(long)p1 - (long)host - 1] = 0x00;
      }
   }

   /* ====== Handle standard address:port ================================ */
   else {
      p1 = strrindex(host,':');
      if(p1 == NULL) {
         p1 = strrindex(host,'!');
      }
      if(p1 != NULL) {
         p1[0] = 0x00;
         strcpy((char*)&port,&p1[1]);
      }
   }

   /* ====== Check port number =========================================== */
   if((sscanf(port,"%d",&portNumber) == 1) &&
      (portNumber < 0) &&
      (portNumber > 65535)) {
      return(false);
   }


   /* ====== Create address structure ==================================== */

   /* ====== Get information for host ==================================== */
   res        = NULL;
   isNumeric  = true;
   isIPv6     = false;
   hostLength = strlen(host);

   memset((char*)&hints,0,sizeof(hints));
   hints.ai_socktype = SOCK_DGRAM;
   hints.ai_family   = AF_UNSPEC;

   for(i = 0;i < hostLength;i++) {
      if(host[i] == ':') {
         isIPv6 = true;
         break;
      }
   }
   if(!isIPv6) {
      for(i = 0;i < hostLength;i++) {
         if(!(isdigit(host[i]) || (host[i] == '.'))) {
            isNumeric = false;
            break;
         }
       }
   }
   if(isNumeric) {
      hints.ai_flags = AI_NUMERICHOST;
   }

   if(getaddrinfo(host,NULL,&hints,&res) != 0) {
      return(false);
   }

   memset((char*)address,0,sizeof(struct sockaddr_storage));
   memcpy((char*)address,res->ai_addr,res->ai_addrlen);

   switch(ipv4address->sin_family) {
      case AF_INET:
         ipv4address->sin_port = htons(portNumber);
       break;
#ifdef HAVE_IPV6
      case AF_INET6:
         ipv6address->sin6_port = htons(portNumber);
       break;
#endif
      default:
         LOG_ERROR
         fprintf(stdlog,"Unsupported address family #%d\n",
                 ((struct sockaddr*)address)->sa_family);
         LOG_END
       break;
   }

   freeaddrinfo(res);
   return(true);
}


/* ###### Print address ################################################## */
void fputaddress(const struct sockaddr* address, const bool port, FILE* fd)
{
   char str[128];

   address2string(address,(char*)&str,sizeof(str),port);
   fputs(str,fd);
}


/* ###### Get IPv4 address scope ######################################### */
static unsigned int scopeIPv4(const uint32_t* address)
{
    uint32_t a;
    uint8_t  b1;
    uint8_t  b2;

    /* Unspecified */
    if(*address == 0x00000000) {
       return(0);
    }

    /* Loopback */
    a = ntohl(*address);
    if((a & 0x7f000000) == 0x7f000000) {
       return(2);
    }

    /* Class A private */
    b1 = (uint8_t)(a >> 24);
    if(b1 == 10) {
       return(6);
    }

    /* Class B private */
    b2 = (uint8_t)((a >> 16) & 0x00ff);
    if((b1 == 172) && (b2 >= 16) && (b2 <= 31)) {
       return(6);
    }

    /* Class C private */
    if((b1 == 192) && (b2 == 168)) {
       return(6);
    }

    if(IN_MULTICAST(ntohl(*address))) {
       return(8);
    }
    return(10);
}


/* ###### Get IPv6 address scope ######################################### */
#ifdef HAVE_IPV6
static unsigned int scopeIPv6(const struct in6_addr* address)
{
   if(IN6_IS_ADDR_V4MAPPED(address)) {
#ifdef LINUX
      return(scopeIPv4(&address->s6_addr32[3]));
#else
      return(scopeIPv4(&address->__u6_addr.__u6_addr32[3]));
#endif
   }
   if(IN6_IS_ADDR_UNSPECIFIED(address)) {
      return(0);
   }
   else if(IN6_IS_ADDR_MC_NODELOCAL(address)) {
      return(1);
   }
   else if(IN6_IS_ADDR_LOOPBACK(address)) {
      return(2);
   }
   else if(IN6_IS_ADDR_MC_LINKLOCAL(address)) {
      return(3);
   }
   else if(IN6_IS_ADDR_LINKLOCAL(address)) {
      return(4);
   }
   else if(IN6_IS_ADDR_MC_SITELOCAL(address)) {
      return(5);
   }
   else if(IN6_IS_ADDR_SITELOCAL(address)) {
      return(6);
   }
   else if(IN6_IS_ADDR_MC_ORGLOCAL(address)) {
      return(7);
   }
   else if(IN6_IS_ADDR_MC_GLOBAL(address)) {
      return(8);
   }
   return(10);
}
#endif


/* ###### Get address scope ############################################## */
unsigned int getScope(const struct sockaddr* address)
{
   if(address->sa_family == AF_INET) {
      return(scopeIPv4((uint32_t*)&((struct sockaddr_in*)address)->sin_addr));
   }
#ifdef HAVE_IPV6
   else if(address->sa_family == AF_INET6) {
      return(scopeIPv6(&((struct sockaddr_in6*)address)->sin6_addr));
   }
#endif
   else {
      LOG_ERROR
      fprintf(stdlog,"Unsupported address family #%d\n",
              ((struct sockaddr*)address)->sa_family);
      LOG_END
   }
   return(0);
}


/* ###### Compare addresses ############################################## */
int addresscmp(const struct sockaddr* a1, const struct sockaddr* a2, const bool port)
{
   uint16_t     p1, p2;
   unsigned int s1, s2;
   uint32_t     x1[4];
   uint32_t     x2[4];
   int          result;

   LOG_VERBOSE5
   fputs("Comparing ",stdlog);
   fputaddress(a1,port,stdlog);
   fputs(" and ",stdlog);
   fputaddress(a2,port,stdlog);
   fputs("\n",stdlog);
   LOG_END

#ifdef HAVE_IPV6
   if( ((a1->sa_family == AF_INET) || (a1->sa_family == AF_INET6)) &&
       ((a2->sa_family == AF_INET) || (a2->sa_family == AF_INET6)) ) {
#else
   if( (a1->sa_family == AF_INET) && (a2->sa_family == AF_INET) ) {
#endif
      s1 = 1000000 - getScope((struct sockaddr*)a1);
      s2 = 1000000 - getScope((struct sockaddr*)a2);
      if(s1 < s2) {
         LOG_VERBOSE5
         fprintf(stdlog,"Result: less-than, s1=%d s2=%d\n",s1,s2);
         LOG_END
         return(-1);
      }
      else if(s1 > s2) {
         LOG_VERBOSE5
         fprintf(stdlog,"Result: greater-than, s1=%d s2=%d\n",s1,s2);
         LOG_END
         return(1);
      }

#ifdef HAVE_IPV6
      if(a1->sa_family == AF_INET6) {
         memcpy((void*)&x1, (void*)&((struct sockaddr_in6*)a1)->sin6_addr, 16);
      }
      else {
#endif
         x1[0] = 0;
         x1[1] = 0;
         x1[2] = htonl(0xffff);
         x1[3] = *((uint32_t*)&((struct sockaddr_in*)a1)->sin_addr);
#ifdef HAVE_IPV6
      }
#endif

#ifdef HAVE_IPV6
      if(a2->sa_family == AF_INET6) {
         memcpy((void*)&x2, (void*)&((struct sockaddr_in6*)a2)->sin6_addr, 16);
      }
      else {
#endif
         x2[0] = 0;
         x2[1] = 0;
         x2[2] = htonl(0xffff);
         x2[3] = *((uint32_t*)&((struct sockaddr_in*)a2)->sin_addr);
#ifdef HAVE_IPV6
      }
#endif

      result = memcmp((void*)&x1,(void*)&x2,16);
      if(result != 0) {
         LOG_VERBOSE5
         if(result < 0) {
            fputs("Result: less-than\n",stdlog);
         }
         else {
            fputs("Result: greater-than\n",stdlog);
         }
         LOG_END
         return(result);
      }

      if(port) {
         p1 = getPort((struct sockaddr*)a1);
         p2 = getPort((struct sockaddr*)a2);
         if(p1 < p2) {
            LOG_VERBOSE5
            fputs("Result: less-than\n",stdlog);
            LOG_END
            return(-1);
         }
         else if(p1 > p2) {
            LOG_VERBOSE5
            fputs("Result: greater-than\n",stdlog);
            LOG_END
            return(1);
         }
      }
      LOG_VERBOSE5
      fputs("Result: equal\n",stdlog);
      LOG_END
      return(0);
   }

   LOG_ERROR
   fprintf(stdlog,"Unsupported address family comparision #%d / #%d\n",a1->sa_family,a2->sa_family);
   LOG_END
   return(0);
}


/* ###### Get port ####################################################### */
uint16_t getPort(struct sockaddr* address)
{
   if(address != NULL) {
      switch(address->sa_family) {
         case AF_INET:
            return(ntohs(((struct sockaddr_in*)address)->sin_port));
          break;
#ifdef HAVE_IPV6
         case AF_INET6:
            return(ntohs(((struct sockaddr_in6*)address)->sin6_port));
          break;
#endif
         default:
            LOG_ERROR
            fprintf(stdlog,"Unsupported address family #%d\n",
                    ((struct sockaddr*)address)->sa_family);
            LOG_END
          break;
      }
   }
   return(0);
}


/* ###### Set port ####################################################### */
bool setPort(struct sockaddr* address, uint16_t port)
{
   if(address != NULL) {
      switch(address->sa_family) {
         case AF_INET:
            ((struct sockaddr_in*)address)->sin_port = htons(port);
            return(true);
          break;
#ifdef HAVE_IPV6
         case AF_INET6:
            ((struct sockaddr_in6*)address)->sin6_port = htons(port);
            return(true);
          break;
#endif
         default:
            LOG_ERROR
            fprintf(stdlog,"Unsupported address family #%d\n",
                    ((struct sockaddr*)address)->sa_family);
            LOG_END
          break;
      }
   }
   return(false);
}


/* ###### Get address family ############################################# */
int getFamily(struct sockaddr* address)
{
   if(address != NULL) {
      return(address->sa_family);
   }
   return(AF_UNSPEC);
}


/* ###### Set non-blocking mode ########################################## */
bool setBlocking(int fd)
{
   int flags = ext_fcntl(fd,F_GETFL,0);
   if(flags != -1) {
      flags &= ~O_NONBLOCK;
      if(ext_fcntl(fd,F_SETFL,flags) == 0) {
         return(true);
      }
   }
   return(false);
}


/* ###### Set blocking mode ############################################## */
bool setNonBlocking(int fd)
{
   int flags = ext_fcntl(fd,F_GETFL,0);
   if(flags != -1) {
      flags |= O_NONBLOCK;
      if(ext_fcntl(fd,F_SETFL,flags) == 0) {
         return(true);
      }
   }
   return(false);
}


/* ###### Get padding #################################################### */
size_t getPadding(const size_t size, const size_t alignment)
{
   size_t padding = alignment - (size % alignment);
   if(padding == alignment) {
      padding = 0;
   }
   return(padding);
}


/* ###### Convert 24-bit value to host byte-order ######################## */
uint32_t ntoh24(const uint32_t value)
{
#if BYTE_ORDER == LITTLE_ENDIAN
   const uint32_t a = ((uint8_t*)&value)[0];
   const uint32_t b = ((uint8_t*)&value)[1];
   const uint32_t c = ((uint8_t*)&value)[2];
   const uint32_t result = (a << 16) | (b << 8) | c;
#elif BYTE_ORDER == BIG_ENDIAN
   const uint32_t result = value;
#else
#error Byte order is not defined!
#endif
   return(result);
}


/* ###### Convert 24-bit value to network byte-order ##################### */
uint32_t hton24(const uint32_t value)
{
   return(ntoh24(value));
}


bool checkIPv6()
{
#ifdef HAVE_IPV6
   int sd = socket(AF_INET6,SOCK_DGRAM,IPPROTO_UDP);
   if(sd >= 0) {
      close(sd);
      return(true);
   }
#endif
   return(false);
}


/* ###### bind()/bindx() wrapper ######################################### */
bool bindplus(int                      sockfd,
              struct sockaddr_storage* addressArray,
              const size_t             addresses)
{
   struct sockaddr_storage anyAddress;
   bool                    autoSelect;
   unsigned short          port;
   unsigned int            i;
   unsigned int            j;
   int                     result;

   if(checkIPv6()) {
      string2address("[::]:0",&anyAddress);
   }
   else {
      string2address("0.0.0.0:0",&anyAddress);
   }

   if((addresses > 0) && (getPort((struct sockaddr*)&addressArray[0]) == 0)) {
      autoSelect = true;
   }
   else {
      autoSelect = false;
   }


   LOG_VERBOSE4
   fputs("Binding socket...\n",stdlog);
   LOG_END

   for(i = 0;i < MAX_AUTOSELECT_TRIALS;i++) {
      if(addresses == 0) {
         port = MIN_AUTOSELECT_PORT + (random16() % (MAX_AUTOSELECT_PORT - MIN_AUTOSELECT_PORT));
         setPort((struct sockaddr*)&anyAddress,port);

         LOG_VERBOSE4
         fprintf(stdlog,"Trying port %d for \"any\" address...\n",port);
         LOG_END

         result = ext_bind(sockfd,(struct sockaddr*)&anyAddress,getSocklen((struct sockaddr*)&anyAddress));
         if(result == 0) {
            LOG_VERBOSE4
            fputs("Successfully bound\n",stdlog);
            LOG_END
            return(true);
         }
         else if(errno == EPROTONOSUPPORT) {
            LOG_VERBOSE4
            logerror("bind() failed");
            LOG_END
            return(false);
         }
      }
      else {
         if(autoSelect) {
            port = MIN_AUTOSELECT_PORT + (random16() % (MAX_AUTOSELECT_PORT - MIN_AUTOSELECT_PORT));
            for(j = 0;j < addresses;j++) {
               setPort((struct sockaddr*)&addressArray[j],port);
            }
            LOG_VERBOSE5
            fprintf(stdlog,"Trying port %d...\n",port);
            LOG_END
         }
         if(addresses == 1) {
            result = ext_bind(sockfd,(struct sockaddr*)&addressArray[0],getSocklen((struct sockaddr*)&addressArray[0]));
         }
         else {
            result = ext_bindx(sockfd,addressArray,addresses,SCTP_BINDX_ADD_ADDR);
         }

         if(result == 0) {
            LOG_VERBOSE4
            fputs("Successfully bound\n",stdlog);
            LOG_END
            return(true);
         }
         else if(errno == EPROTONOSUPPORT) {
            LOG_VERBOSE4
            logerror("bind() failed");
            LOG_END
            return(false);
         }
         if(!autoSelect) {
            return(false);
         }
      }
   }
   return(result);
}


/* ###### sendmsg() wrapper ############################################## */
int sendtoplus(int                 sockfd,
               void*               buffer,
               const size_t        length,
               const int           flags,
               struct sockaddr*    to,
               const socklen_t     tolen,
               const uint32_t      ppid,
               const sctp_assoc_t  assocID,
               const uint16_t streamID,
               const uint32_t      timeToLive,
               const card64        timeout)
{
   struct sctp_sndrcvinfo* sri;
   struct iovec    iov = { (char*)buffer, length };
   struct cmsghdr* cmsg;
   size_t          cmsglen = CMSG_SPACE(sizeof(struct sctp_sndrcvinfo));
   char            cbuf[CMSG_SPACE(sizeof(struct sctp_sndrcvinfo))];
   struct msghdr msg = {
#ifdef __APPLE__
      (char*)to,
#else
      to,
#endif
      tolen,
      &iov, 1,
      cbuf, cmsglen,
      flags
   };
   struct timeval selectTimeout;
   fd_set         fdset;
   int            result;
   int            cc;

   cmsg = (struct cmsghdr*)CMSG_FIRSTHDR(&msg);
   cmsg->cmsg_len   = CMSG_LEN(sizeof(struct sctp_sndrcvinfo));
   cmsg->cmsg_level = IPPROTO_SCTP;
   cmsg->cmsg_type  = SCTP_SNDRCV;

   sri = (struct sctp_sndrcvinfo*)CMSG_DATA(cmsg);
   sri->sinfo_assoc_id   = assocID;
   sri->sinfo_stream     = streamID;
   sri->sinfo_ppid       = ppid;
   sri->sinfo_flags      = flags;
   sri->sinfo_ssn        = 0;
   sri->sinfo_tsn        = 0;
   sri->sinfo_context    = 0;
   sri->sinfo_timetolive = timeToLive;

   LOG_VERBOSE4
   fprintf(stdlog,"sendmsg(%d,%u bytes)...\n",sockfd,(unsigned int)length);
   LOG_END

   setNonBlocking(sockfd);
   cc = ext_sendmsg(sockfd,&msg,flags);
   if((timeout > 0) && ((cc < 0) && (errno == EWOULDBLOCK))) {
      LOG_VERBOSE4
      fprintf(stdlog,"sendmsg(%d) would block, waiting with timeout %Ld [s]...\n",sockfd,timeout);
      LOG_END

      FD_ZERO(&fdset);
      FD_SET(sockfd,&fdset);
      selectTimeout.tv_sec  = timeout / 1000000;
      selectTimeout.tv_usec = timeout % 1000000;
      result = ext_select(sockfd + 1,
                          NULL, &fdset, NULL,
                          &selectTimeout);
      if(result > 0) {
         LOG_VERBOSE4
         fprintf(stdlog,"retrying sendmsg(%d, %u bytes)...\n",
                 sockfd, (unsigned int)iov.iov_len);
         LOG_END
         cc = ext_sendmsg(sockfd,&msg,flags);
      }
      else {
         cc    = -1;
         errno = EWOULDBLOCK;
         LOG_VERBOSE5
         fprintf(stdlog,"sendmsg(%d) timed out\n",sockfd);
         LOG_END
      }
   }

   LOG_VERBOSE4
   fprintf(stdlog,"sendmsg(%d) result=%d; %s\n",
           sockfd, cc, strerror(errno));
   LOG_END

   return(cc);
}


/* ###### recvmsg() wrapper ############################################## */
int recvfromplus(int              sockfd,
                 void*            buffer,
                 size_t           length,
                 int              flags,
                 struct sockaddr* from,
                 socklen_t*       fromlen,
                 uint32_t*        ppid,
                 sctp_assoc_t*    assocID,
                 uint16_t*   streamID,
                 const card64     timeout)
{
   struct sctp_sndrcvinfo* sri;
   struct iovec    iov = { (char*)buffer, length };
   struct cmsghdr* cmsg;
   size_t          cmsglen = CMSG_SPACE(sizeof(struct sctp_sndrcvinfo));
   char            cbuf[CMSG_SPACE(sizeof(struct sctp_sndrcvinfo))];
   struct msghdr msg = {
#ifdef __APPLE__
      (char*)from,
#else
      from,
#endif
      (fromlen != NULL) ? *fromlen : 0,
      &iov, 1,
      cbuf, cmsglen,
      flags
   };
   struct timeval selectTimeout;
   fd_set         fdset;
   int            result;
   int            cc;

   if(ppid     != NULL) *ppid     = 0;
   if(streamID != NULL) *streamID = 0;
   if(assocID  != NULL) *assocID  = 0;

   LOG_VERBOSE5
   fprintf(stdlog,"recvmsg(%d, %u bytes)...\n",
           sockfd, (unsigned int)iov.iov_len);
   LOG_END

   setNonBlocking(sockfd);
   cc = ext_recvmsg(sockfd,&msg,flags);
   if((timeout > 0) && ((cc < 0) && (errno == EWOULDBLOCK))) {
      LOG_VERBOSE5
      fprintf(stdlog,"recvmsg(%d) would block, waiting with timeout %Ld [s]...\n",sockfd,timeout);
      LOG_END

      FD_ZERO(&fdset);
      FD_SET(sockfd,&fdset);
      selectTimeout.tv_sec  = timeout / 1000000;
      selectTimeout.tv_usec = timeout % 1000000;
      result = ext_select(sockfd + 1,
                          &fdset, NULL, NULL,
                          &selectTimeout);
      if((result > 0) && FD_ISSET(sockfd, &fdset)) {
         LOG_VERBOSE5
         fprintf(stdlog,"retrying recvmsg(%d, %u bytes)...\n",
                 sockfd, (unsigned int)iov.iov_len);
         LOG_END
         cc = ext_recvmsg(sockfd,&msg,flags);
      }
      else {
         LOG_VERBOSE5
         fprintf(stdlog,"recvmsg(%d) timed out\n", sockfd);
         LOG_END
         cc    = -1;
         errno = EWOULDBLOCK;
      }
   }

   LOG_VERBOSE4
   fprintf(stdlog,"recvmsg(%d) result=%d data=%u/%u control=%u; %s\n",
           sockfd, cc, (unsigned int)iov.iov_len, (unsigned int)length, (unsigned int)msg.msg_controllen, strerror(errno));
   LOG_END

   if(cc < 0) {
      return(cc);
   }

   if((msg.msg_control != NULL) && (msg.msg_controllen > 0)) {
      cmsg = (struct cmsghdr*)CMSG_FIRSTHDR(&msg);
      if((cmsg != NULL) &&
         (cmsg->cmsg_len   == CMSG_LEN(sizeof(struct sctp_sndrcvinfo))) &&
         (cmsg->cmsg_level == IPPROTO_SCTP)                             &&
         (cmsg->cmsg_type  == SCTP_SNDRCV)) {
         sri = (struct sctp_sndrcvinfo*)CMSG_DATA(cmsg);
         if(ppid     != NULL) *ppid     = sri->sinfo_ppid;
         if(streamID != NULL) *streamID = sri->sinfo_stream;
         if(assocID  != NULL) *assocID  = sri->sinfo_assoc_id;
      }
   }
   if(fromlen != NULL) {
      *fromlen = msg.msg_namelen;
   }
   return(cc);
}


/* ###### Multicast group management ##################################### */
bool multicastGroupMgt(int                      sockfd,
                       struct sockaddr_storage* address,
                       const char*              interface,
                       const bool               add)
{
   struct ip_mreq   mreq;
   struct ifreq     ifr;
#ifdef HAVE_IPV6
   struct ipv6_mreq mreq6;
#endif

   if(((struct sockaddr*)address)->sa_family == AF_INET) {
      mreq.imr_multiaddr = ((struct sockaddr_in*)address)->sin_addr;
      if(interface != NULL) {
         strcpy(ifr.ifr_name,interface);
         if(ext_ioctl(sockfd,SIOCGIFADDR,&ifr) != 0) {
            return(false);
         }
         mreq.imr_interface = ((struct sockaddr_in*)&ifr.ifr_addr)->sin_addr;
      }
      else {
         memset((char*)&mreq.imr_interface,0,sizeof(mreq.imr_interface));
      }
      return(ext_setsockopt(sockfd,IPPROTO_IP,
                            add ? IP_ADD_MEMBERSHIP : IP_DROP_MEMBERSHIP,
                            &mreq, sizeof(mreq)) == 0);
   }
#ifdef HAVE_IPV6
   else if(((struct sockaddr*)address)->sa_family == AF_INET6) {
      memcpy((char*)&mreq6.ipv6mr_multiaddr,
             (char*)&((struct sockaddr_in6*)address)->sin6_addr,
             sizeof(((struct sockaddr_in6*)address)->sin6_addr));
      if(interface != NULL) {
         mreq6.ipv6mr_interface = if_nametoindex(interface);
      }
      else {
         mreq6.ipv6mr_interface = 0;
      }
      return(ext_setsockopt(sockfd,IPPROTO_IPV6,
                            add ? IPV6_JOIN_GROUP : IPV6_LEAVE_GROUP,
                            &mreq6, sizeof(mreq6)) == 0);
   }
#endif
   return(false);
}


/* ###### Get socklen for given address ################################## */
size_t getSocklen(struct sockaddr* address)
{
   switch(address->sa_family) {
      case AF_INET:
         return(sizeof(struct sockaddr_in));
       break;
      case AF_INET6:
         return(sizeof(struct sockaddr_in6));
       break;
      default:
         LOG_ERROR
         fprintf(stdlog,"Unsupported address family #%d\n",
                 ((struct sockaddr*)address)->sa_family);
         LOG_END
         return(sizeof(struct sockaddr));
       break;
   }
}
