/*
 *  $Id: rsplib.c,v 1.14 2002/08/22 09:37:25 dreibh Exp $
 *
 * RSerPool implementation.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de/rserpool.html
 * which should be used for any discussion related to this implementation.
 *
 * Contact: rsplib-discussion@sctp.de
 *          dreibh@exp-math.uni-essen.de
 *
 * Purpose: RSerPool Library
 *
 */


#include "tdtypes.h"
#include "rsplib.h"
#include "asapinstance.h"
#include "utilities.h"
#include "netutilities.h"

#include <ext_socket.h>
#ifdef _THREAD_SAFE
#include <pthread.h>
#endif


#define MAX_ADDRESSES_PER_ENDPOINT 128


static struct Dispatcher*   gDispatcher   = NULL;
static struct ASAPInstance* gAsapInstance = NULL;
static enum ASAPError       gLastError    = ASAP_Okay;
#ifdef _THREAD_SAFE
static pthread_mutex_t      gMutex;
static pthread_t            gMutexOwner;
static unsigned int         gMutexRecursionLevel;
#endif


#ifdef _THREAD_SAFE
/* ###### Lock mutex ###################################################### */
static void lock(struct Dispatcher* dispatcher, void* userData)
{
   if(!pthread_equal(gMutexOwner,pthread_self())) {
      pthread_mutex_lock(&gMutex);
      gMutexOwner = pthread_self();
   }
   gMutexRecursionLevel++;
}


/* ###### Unlock mutex #################################################### */
static void unlock(struct Dispatcher* dispatcher, void* userData)
{
   if(gMutexRecursionLevel == 0) {
      LOG_ERROR
      fputs("Mutex is already unlocked!\n",stdlog);
      LOG_END
      exit(1);
   }
   if(pthread_equal(gMutexOwner,pthread_self())) {
      gMutexRecursionLevel--;
      if(gMutexRecursionLevel == 0) {
         gMutexOwner = 0;
         pthread_mutex_unlock(&gMutex);
      }
   }
   else {
      LOG_ERROR
      fputs("Mutex is not owned!\n",stdlog);
      LOG_END
   }
}
#endif


/* ###### Initialize library ############################################# */
int rspInitialize(struct TagItem* tags)
{
#ifdef _THREAD_SAFE
   gMutexOwner          = 0;
   gMutexRecursionLevel = 0;
   pthread_mutex_init(&gMutex,NULL);
#endif

   gDispatcher = dispatcherNew(
#ifdef _THREAD_SAFE
                    lock,
                    unlock,
                    NULL
#else
                    dispatcherDefaultLock,
                    dispatcherDefaultUnlock,
                    NULL
#endif
   );

   if(gDispatcher) {
      gAsapInstance = asapNew(gDispatcher);
      if(gAsapInstance) {
         return(0);
      }
      else {
         dispatcherDelete(gDispatcher);
         gDispatcher = NULL;
      }
   }

   return(ENOMEM);
}


/* ###### Clean-up library ############################################### */
void rspCleanUp()
{
   if(gAsapInstance) {
      asapDelete(gAsapInstance);
      dispatcherDelete(gDispatcher);
#ifdef _THREAD_SAFE
      pthread_mutex_destroy(&gMutex);
#endif
      gAsapInstance = NULL;
      gDispatcher   = NULL;

      /* Give sctplib some time to cleanly shut down all associations */
      usleep(250000);
   }
}


/* ###### Get last error ################################################# */
unsigned int rspGetLastError()
{
   return((unsigned int)gLastError);
}


/* ###### Get last error description ##################################### */
const char* rspGetLastErrorDescription()
{
   return(asapErrorGetDescription(gLastError));
}


/* ###### Register pool element ########################################## */
uint32_t rspRegister(const char*                       poolHandle,
                     const size_t                      poolHandleSize,
                     const struct EndpointAddressInfo* endpointAddressInfo,
                     struct TagItem*                   tags)
{
   struct PoolHandle*                myPoolHandle;
   struct PoolElement*               myPoolElement;
   struct PoolPolicy*                myPolicy;
   PoolElementIdentifier             myIdentifier = UNDEFINED_POOL_ELEMENT_IDENTIFIER;
   struct TransportAddress*          transportAddress;
   const struct EndpointAddressInfo* endpointAddress;
   size_t                            addresses;
   struct sockaddr_storage           addressArray[MAX_ADDRESSES_PER_ENDPOINT];
   char*                             ptr;
   int                               error = 0;
   size_t                            i;

   gLastError = ASAP_Okay;
   if(gAsapInstance) {
      myPoolHandle = poolHandleNew(poolHandle,poolHandleSize);
      if(myPoolHandle) {
         myIdentifier = endpointAddressInfo->ai_pe_id;
         if(myIdentifier == UNDEFINED_POOL_ELEMENT_IDENTIFIER) {
            myIdentifier = getPoolElementIdentifier();
         }
         myPolicy = poolPolicyNew((uint8_t)tagListGetData(tags,TAG_PoolPolicy_Type,TAGDATA_PoolPolicy_Type_RoundRobin));
         if(myPolicy) {
            myPolicy->Weight = (unsigned int)tagListGetData(tags,TAG_PoolPolicy_Parameter_Weight,1);
            myPolicy->Load   = (unsigned int)tagListGetData(tags,TAG_PoolPolicy_Parameter_Load,0);
            myPoolElement = poolElementNew(myIdentifier, myPolicy);
            if(myPoolElement != NULL) {
               endpointAddress = endpointAddressInfo;
               while(endpointAddress != NULL) {
                  addresses = endpointAddress->ai_addrs;
                  if(addresses > MAX_ADDRESSES_PER_ENDPOINT) {
                     error = EINVAL;
                     LOG_ERROR
                     fprintf(stdlog,"Too many addresses: %d\n",addresses);
                     LOG_END
                     break;
                  }
                  if(endpointAddress->ai_addrlen > sizeof(struct sockaddr_storage)) {
                     error = EINVAL;
                     LOG_ERROR
                     fprintf(stdlog,"Bad address length: %d\n",endpointAddress->ai_addrlen);
                     LOG_END
                     break;
                  }

                  ptr = (char*)endpointAddress->ai_addr;
                  for(i = 0;i < addresses;i++) {
                     memcpy((void*)&addressArray[i],
                            (void*)ptr,
                            endpointAddress->ai_addrlen);
                     ptr = (char*)((long)ptr + (long)endpointAddress->ai_addrlen);
                  }

                  transportAddress = transportAddressNew(endpointAddress->ai_protocol,
                                                         getPort((struct sockaddr*)endpointAddress->ai_addr),
                                                         (struct sockaddr_storage*)&addressArray,
                                                         addresses);
                  if(transportAddress == NULL) {
                     error = ENOMEM;
                     break;
                  }

                  poolElementAddTransportAddress(myPoolElement,transportAddress);

                  endpointAddress = endpointAddress->ai_next;
               }


               if(error == 0) {
                  LOG_ACTION
                  fputs("Trying to register ",stdlog);
                  poolElementPrint(myPoolElement,stdlog);
                  LOG_END

                  gLastError = asapRegister(gAsapInstance,myPoolHandle,myPoolElement);
                  if(gLastError != ASAP_Okay) {
                     switch(gLastError) {
                        default:
                           error = EIO;
                         break;
                     }
                  }
               }

               poolElementDelete(myPoolElement);
            }
            else {
               error = ENOMEM;
            }

            poolPolicyDelete(myPolicy);
         }
         else {
            error = ENOMEM;
         }

         poolHandleDelete(myPoolHandle);
      }
      else {
         error = ENOMEM;
      }
   }
   else {
      error = EPERM;
      LOG_ERROR
      fputs("rsplib is not initialized\n",stdlog);
      LOG_END
   }

   if(error) {
      errno = error;
      return(0);
   }
   return(myIdentifier);
}


/* ###### Deregister pool element ######################################## */
int rspDeregister(const char*     poolHandle,
                  const size_t    poolHandleSize,
                  const uint32_t  identifier,
                  struct TagItem* tags)
{
   struct PoolHandle* myPoolHandle;
   int                error;

   error = 0;
   gLastError = ASAP_Okay;
   if(gAsapInstance) {
      myPoolHandle = poolHandleNew(poolHandle,poolHandleSize);
      if(myPoolHandle) {
         gLastError = asapDeregister(gAsapInstance,myPoolHandle,identifier);
         if(gLastError != ASAP_Okay) {
            switch(gLastError) {
               default:
                  error = EIO;
                break;
            }
         }
         poolHandleDelete(myPoolHandle);
      }
      else {
         error = ENOMEM;
      }
   }
   else {
      error = EPERM;
      LOG_ERROR
      fputs("rsplib is not initialized\n",stdlog);
      LOG_END
   }

   errno = error;
   return(error);
}


/* ###### Name resolution ################################################ */
int rspNameResolution(const char*                  poolHandle,
                      const size_t                 poolHandleSize,
                      struct EndpointAddressInfo** endpointAddressInfo,
                      struct TagItem*              tags)
{
   struct PoolHandle*          myPoolHandle;
   struct PoolElement*         poolElement;
   struct TransportAddress*    transportAddress;
   struct EndpointAddressInfo* endpointAddress;
   GList*                      list;
   int                         error;
   size_t                      i;
   char*                       ptr;

   error                = 0;
   gLastError           = ASAP_Okay;
   *endpointAddressInfo = NULL;
   if(gAsapInstance) {
      myPoolHandle = poolHandleNew(poolHandle,poolHandleSize);
      if(myPoolHandle) {
         poolElement = asapSelectPoolElement(gAsapInstance,myPoolHandle,&gLastError);
         if(poolElement != NULL) {
            list = g_list_last(poolElement->TransportAddressList);
            while(list != NULL) {
               transportAddress = (struct TransportAddress*)list->data;
               if(transportAddress->Addresses > 0) {
                  endpointAddress = (struct EndpointAddressInfo*)malloc(sizeof(struct EndpointAddressInfo));
                  if(endpointAddress != NULL) {
                     endpointAddress->ai_next     = *endpointAddressInfo;
                     endpointAddress->ai_pe_id    = poolElement->Identifier;
                     endpointAddress->ai_family   = transportAddress->AddressArray[0].sa.sa_family;
                     endpointAddress->ai_protocol = transportAddress->Protocol;
                     switch(transportAddress->Protocol) {
                        case IPPROTO_SCTP:
                        case IPPROTO_TCP:
                           endpointAddress->ai_socktype = SOCK_STREAM;
                         break;
                        default:
                           endpointAddress->ai_socktype = SOCK_DGRAM;
                         break;
                     }
                     endpointAddress->ai_addrlen = sizeof(struct sockaddr_storage);
                     endpointAddress->ai_addrs   = transportAddress->Addresses;
                     endpointAddress->ai_addr    = (struct sockaddr_storage*)malloc(endpointAddress->ai_addrs * sizeof(struct sockaddr_storage));
                     if(endpointAddress->ai_addr != NULL) {
                        ptr = (char*)endpointAddress->ai_addr;
                        for(i = 0;i < transportAddress->Addresses;i++) {
                           memcpy((void*)ptr, (void*)&transportAddress->AddressArray[i], sizeof(union sockaddr_union));
                           ptr = (char*)((long)ptr + (long)sizeof(struct sockaddr_storage));
                        }

                        *endpointAddressInfo = endpointAddress;
                     }
                     else {
                        free(endpointAddress);
                        endpointAddress = NULL;
                     }
                  }
               }

               list = g_list_previous(list);
            }

            poolElementDelete(poolElement);
         }
         else {
            error = ENOENT;
         }
         poolHandleDelete(myPoolHandle);
      }
      else {
         error = ENOMEM;
      }
   }
   else {
      error = EPERM;
      LOG_ERROR
      fputs("rsplib is not initialized\n",stdlog);
      LOG_END
   }

   errno = error;
   return(error);
}


/* ###### Free endpoint address array #################################### */
void rspFreeEndpointAddressArray(struct EndpointAddressInfo* endpointAddressInfo)
{
   struct EndpointAddressInfo* next;

   while(endpointAddressInfo != NULL) {
      next = endpointAddressInfo->ai_next;

      if(endpointAddressInfo->ai_addr) {
         free(endpointAddressInfo->ai_addr);
         endpointAddressInfo->ai_addr = NULL;
      }
      free(endpointAddressInfo);

      endpointAddressInfo = next;
   }
}


/* ###### Report pool element failure #################################### */
void rspFailure(const char*     poolHandle,
                const size_t    poolHandleSize,
                const uint32_t  identifier,
                struct TagItem* tags)
{
   struct PoolHandle* myPoolHandle;

   if(gAsapInstance) {
       myPoolHandle = poolHandleNew(poolHandle,poolHandleSize);
       if(myPoolHandle) {
          asapFailure(gAsapInstance,myPoolHandle,identifier);
          poolHandleDelete(myPoolHandle);
       }
   }
}


/* ###### select() wrapper ############################################### */
int rspSelect(int             n,
              fd_set*         readfds,
              fd_set*         writefds,
              fd_set*         exceptfds,
              struct timeval* timeout)
{
   fd_set myreadfds;
   fd_set mywritefds;
   fd_set myexceptfds;
   int    myn;
   int    i;
   card64 asapTimeout;
   card64 userTimeout;
   card64 newTimeout;
   int    result;

   if(gDispatcher) {
      userTimeout = ((card64)timeout->tv_sec * 1000000) + (card64)timeout->tv_usec;
      dispatcherGetSelectParameters(gDispatcher, &myn, &myreadfds, &mywritefds, &myexceptfds, timeout);
      asapTimeout = ((card64)timeout->tv_sec * 1000000) + (card64)timeout->tv_usec;
      newTimeout  = min(userTimeout,asapTimeout);
      timeout->tv_sec  = newTimeout / 1000000;
      timeout->tv_usec = newTimeout % 1000000;

      if(readfds) {
         for(i = 0;i < n;i++) {
            if(FD_ISSET(i,readfds)) {
               FD_SET(i,&myreadfds);
            }
         }
      }
      if(writefds) {
         for(i = 0;i < n;i++) {
            if(FD_ISSET(i,writefds)) {
               FD_SET(i,&mywritefds);
            }
         }
      }
      if(exceptfds) {
         for(i = 0;i < n;i++) {
            if(FD_ISSET(i,exceptfds)) {
               FD_SET(i,&myexceptfds);
            }
         }
      }
      myn = max(myn,n);

      result = ext_select(myn, &myreadfds, &mywritefds, &myexceptfds, timeout);

      dispatcherHandleSelectResult(gDispatcher, result, &myreadfds, &mywritefds, &myexceptfds);
      if(result > 0) {
         result = 0;
         if(readfds) {
            for(i = 0;i < n;i++) {
               if(FD_ISSET(i,&myreadfds)) {
                  FD_SET(i,readfds);
                  result++;
               }
               else {
                  FD_CLR(i,readfds);
               }
            }
         }
         if(writefds) {
            for(i = 0;i < n;i++) {
               if(FD_ISSET(i,&mywritefds)) {
                  FD_SET(i,writefds);
                  result++;
               }
               else {
                  FD_CLR(i,writefds);
               }
            }
         }
         if(exceptfds) {
            for(i = 0;i < n;i++) {
               if(FD_ISSET(i,&myexceptfds)) {
                  FD_SET(i,exceptfds);
                  result++;
               }
               else {
                  FD_CLR(i,exceptfds);
               }
            }
         }
      }
   }
   else {
      result = ext_select(n, readfds, writefds, exceptfds, timeout);
   }

   return(result);
}
